<?php
/**
 * Система обновлений для плагина OK Auto Post
 */

// Защита от прямого доступа
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Класс для управления обновлениями плагина
 */
class OkAutoPostUpdater {
    
    private $update_url = 'https://gaagaa.ru/my_codes/WordPress_plugin/OK_Auto_Post/updates/version.json';
    private $plugin_slug;
    private $plugin_basename;
    private $current_version;
    
    public function __construct() {
        $this->current_version = OK_AUTO_POST_VERSION;
        $this->plugin_slug = 'ok-auto-post';
        $this->plugin_basename = plugin_basename(OK_AUTO_POST_PLUGIN_FILE);
        
        // Регистрируем хуки для обновлений
        add_filter('pre_set_site_transient_update_plugins', array($this, 'check_update'));
        add_filter('plugins_api', array($this, 'plugins_api_handler'), 10, 3);
        add_filter('upgrader_post_install', array($this, 'post_install'), 10, 3);
        
        // Добавляем информацию о плагине для проверки обновлений
        add_filter('site_transient_update_plugins', array($this, 'inject_update'));
    }
    
    /**
     * Проверка обновлений через WordPress
     */
    public function check_update($transient) {
        // Если нет данных о проверенных плагинах, выходим
        if (empty($transient->checked)) {
            return $transient;
        }
        
        // Получаем информацию об обновлениях
        $remote_data = $this->get_remote_info();
        
        // Если есть обновление и версия выше текущей
        if ($remote_data && version_compare($this->current_version, $remote_data['version'], '<')) {
            $obj = new stdClass();
            $obj->slug = $this->plugin_slug;
            $obj->plugin = $this->plugin_basename;
            $obj->new_version = $remote_data['version'];
            $obj->tested = $remote_data['tested'] ?? '6.4';
            $obj->requires = $remote_data['requires'] ?? '5.0';
            $obj->requires_php = $remote_data['requires_php'] ?? '7.4';
            $obj->package = $remote_data['package'] ?? $remote_data['download_url'];
            $obj->url = $remote_data['url'] ?? 'https://gaagaa.ru/my_codes/WordPress_plugin/OK_Auto_Post/';
            $obj->last_updated = $remote_data['last_updated'] ?? date('Y-m-d');
            
            $transient->response[$this->plugin_basename] = $obj;
        }
        
        return $transient;
    }
    
    /**
     * Инъекция данных об обновлении в трансмит
     */
    public function inject_update($transient) {
        return $this->check_update($transient);
    }
    
    /**
     * Обработка информации о плагине для окна обновлений
     */
    public function plugins_api_handler($result, $action, $args) {
        if ($action !== 'plugin_information') {
            return $result;
        }
        
        if (!isset($args->slug) || $args->slug !== $this->plugin_slug) {
            return $result;
        }
        
        $remote_data = $this->get_remote_info();
        
        if (!$remote_data) {
            return $result;
        }
        
        $result = new stdClass();
        $result->name = 'OK Auto Post';
        $result->slug = $this->plugin_slug;
        $result->version = $remote_data['version'];
        $result->tested = $remote_data['tested'] ?? '6.4';
        $result->requires = $remote_data['requires'] ?? '5.0';
        $result->requires_php = $remote_data['requires_php'] ?? '7.4';
        $result->last_updated = $remote_data['last_updated'] ?? date('Y-m-d');
        $result->download_link = $remote_data['download_url'] ?? $remote_data['package'];
        $result->url = $remote_data['url'] ?? 'https://gaagaa.ru/my_codes/WordPress_plugin/OK_Auto_Post/';
        $result->author = '<a href="https://gaagaa.ru/">Громов А. А.</a>';
        $result->homepage = 'https://gaagaa.ru/my_codes/WordPress_plugin/OK_Auto_Post/';
        $result->sections = array(
            'description' => $remote_data['sections']['description'] ?? 'Автоматический постинг в группы Одноклассников.',
            'changelog' => $remote_data['sections']['changelog'] ?? 'Смотрите полную историю изменений на сайте плагина.'
        );
        
        return $result;
    }
    
    /**
     * Действия после установки обновления
     */
    public function post_install($response, $hook_extra, $result) {
        global $wp_filesystem;
        
        $install_directory = plugin_dir_path(OK_AUTO_POST_PLUGIN_FILE);
        $wp_filesystem->move($result['destination'], $install_directory);
        $result['destination'] = $install_directory;
        
        if (is_plugin_active($this->plugin_basename)) {
            activate_plugin($this->plugin_basename);
        }
        
        return $result;
    }
    
    /**
     * Получение информации с сервера обновлений
     */
    public function get_remote_info() {
        $remote_data = get_transient('ok_auto_post_remote_data');
        
        if (false === $remote_data) {
            $response = wp_remote_get($this->update_url, array(
                'timeout' => 10,
                'sslverify' => false,
                'headers' => array(
                    'Accept' => 'application/json'
                )
            ));
            
            if (is_wp_error($response)) {
                return false;
            }
            
            $status_code = wp_remote_retrieve_response_code($response);
            if ($status_code !== 200) {
                return false;
            }
            
            $body = wp_remote_retrieve_body($response);
            $remote_data = json_decode($body, true);
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                return false;
            }
            
            if (!isset($remote_data['version']) || !isset($remote_data['download_url'])) {
                return false;
            }
            
            set_transient('ok_auto_post_remote_data', $remote_data, 12 * HOUR_IN_SECONDS);
        }
        
        return $remote_data;
    }
    
    /**
     * Принудительная проверка обновлений
     */
    public function force_check() {
        delete_transient('ok_auto_post_remote_data');
        delete_site_transient('update_plugins');
        wp_clean_plugins_cache();
        return $this->get_remote_info();
    }
}