<?php
/**
 * Класс для работы с системой лицензирования
 */

if (!defined('ABSPATH')) {
    exit;
}

class OkAutoPostLicensing {
    
    /**
     * Проверка формата лицензионного ключа
     */
    public static function validate_license_format($license_key) {
        return preg_match('/^OK-PRO-[A-Z0-9]{4}-[A-Z0-9]{4}-[A-Z0-9]{4}-[A-Z0-9]{4}-[A-Z0-9]{4}$/i', $license_key);
    }
    
    /**
     * Активация лицензии на сервере
     */
    public static function activate_on_server($license_key, $email) {
        $site_url = home_url();
        $site_name = get_bloginfo('name');
        
        $response = wp_remote_post(OKAP_VERIFY_LICENSE_ENDPOINT, array(
            'timeout' => 30,
            'sslverify' => false,
            'headers' => array(
                'Content-Type' => 'application/json',
            ),
            'body' => json_encode(array(
                'license_key' => $license_key,
                'domain' => $site_url,
                'action' => 'activate',
                'site_name' => $site_name,
                'email' => $email
            ))
        ));
        
        if (is_wp_error($response)) {
            throw new Exception('Ошибка подключения к серверу лицензий: ' . $response->get_error_message());
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if (empty($data)) {
            throw new Exception('Неверный ответ от сервера лицензий');
        }
        
        if (!$data['success']) {
            throw new Exception($data['error'] ?? 'Ошибка активации лицензии');
        }
        
        return $data['license'] ?? array();
    }
    
    /**
     * Деактивация лицензии на сервере
     */
    public static function deactivate_on_server($license_key, $site_url) {
        $response = wp_remote_post(OKAP_VERIFY_LICENSE_ENDPOINT, array(
            'timeout' => 30,
            'sslverify' => false,
            'headers' => array(
                'Content-Type' => 'application/json',
            ),
            'body' => json_encode(array(
                'license_key' => $license_key,
                'domain' => $site_url,
                'action' => 'deactivate'
            ))
        ));
        
        if (is_wp_error($response)) {
            // При деактивации не бросаем исключение, просто логируем
            error_log('Ошибка деактивации лицензии: ' . $response->get_error_message());
            return false;
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if (empty($data) || !$data['success']) {
            return false;
        }
        
        return true;
    }
    
    /**
     * Проверка статуса лицензии
     */
    public static function check_license_status($license_key = null, $force = false) {
        if (!$license_key) {
            $license_key = get_option(OKAP_LICENSE_KEY_OPTION);
        }
        
        if (!$license_key) {
            return false;
        }
        
        $last_check = get_option(OKAP_LICENSE_LAST_CHECK_OPTION);
        
        // Проверяем не чаще чем раз в 12 часов, если не принудительно
        if (!$force && $last_check && (time() - $last_check) < 12 * HOUR_IN_SECONDS) {
            return get_option(OKAP_LICENSE_STATUS_OPTION) === 'valid';
        }
        
        $site_url = home_url();
        
        $response = wp_remote_post(OKAP_VERIFY_LICENSE_ENDPOINT, array(
            'timeout' => 30,
            'sslverify' => false,
            'headers' => array(
                'Content-Type' => 'application/json',
            ),
            'body' => json_encode(array(
                'license_key' => $license_key,
                'domain' => $site_url,
                'action' => 'verify'
            ))
        ));
        
        if (is_wp_error($response)) {
            // Сохраняем текущий статус при ошибке сети
            return get_option(OKAP_LICENSE_STATUS_OPTION) === 'valid';
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if (empty($data)) {
            return false;
        }
        
        if ($data['success']) {
            update_option(OKAP_LICENSE_STATUS_OPTION, 'valid');
            update_option(OKAP_LICENSE_DATA_OPTION, $data['license'] ?? array());
            update_option(OKAP_LICENSE_LAST_CHECK_OPTION, time());
            delete_option(OKAP_LICENSE_ERROR_OPTION);
            return true;
        } else {
            update_option(OKAP_LICENSE_STATUS_OPTION, 'invalid');
            update_option(OKAP_LICENSE_ERROR_OPTION, $data['error'] ?? 'Лицензия недействительна');
            return false;
        }
    }
    
    /**
     * Проверка, активна ли лицензия
     */
    public static function is_license_valid() {
        $status = get_option(OKAP_LICENSE_STATUS_OPTION);
        $license_key = get_option(OKAP_LICENSE_KEY_OPTION);
        
        if ($status !== 'valid' || !$license_key) {
            return false;
        }
        
        $license_data = get_option(OKAP_LICENSE_DATA_OPTION, array());
        
        // Проверяем срок действия лицензии
        if (isset($license_data['expires_at'])) {
            $expires_date = strtotime($license_data['expires_at']);
            if ($expires_date && time() > $expires_date) {
                update_option(OKAP_LICENSE_STATUS_OPTION, 'expired');
                update_option(OKAP_LICENSE_ERROR_OPTION, 'Срок действия лицензии истек');
                return false;
            }
        }
        
        return true;
    }
    
    /**
     * Получение данных лицензии
     */
    public static function get_license_data() {
        return get_option(OKAP_LICENSE_DATA_OPTION, array());
    }
    
    /**
     * Получение лицензионного ключа
     */
    public static function get_license_key() {
        return get_option(OKAP_LICENSE_KEY_OPTION);
    }
    
    /**
     * Получение последней ошибки лицензии
     */
    public static function get_license_error() {
        return get_option(OKAP_LICENSE_ERROR_OPTION, '');
    }
    
    /**
     * Сохранение лицензионного ключа
     */
    public static function save_license_key($license_key, $email) {
        update_option(OKAP_LICENSE_KEY_OPTION, $license_key);
        
        // Обновляем опции плагина
        $options = get_option('ok_auto_post_options', array());
        $options['license_key'] = $license_key;
        $options['license_email'] = $email;
        update_option('ok_auto_post_options', $options);
    }
    
    /**
     * Удаление лицензионного ключа
     */
    public static function remove_license_key() {
        delete_option(OKAP_LICENSE_KEY_OPTION);
        delete_option(OKAP_LICENSE_STATUS_OPTION);
        delete_option(OKAP_LICENSE_DATA_OPTION);
        delete_option(OKAP_LICENSE_LAST_CHECK_OPTION);
        delete_option(OKAP_LICENSE_ERROR_OPTION);
        
        // Обновляем опции плагина
        $options = get_option('ok_auto_post_options', array());
        unset($options['license_key']);
        unset($options['license_email']);
        unset($options['license_status']);
        unset($options['license_expires']);
        update_option('ok_auto_post_options', $options);
    }
}