<?php

class OkAutoPostCore {
    
    private $options;
    private $auto_posting_enabled = true;
    private static $hooks_registered = false;
    
    public function __construct() {
        $this->options = get_option('ok_auto_post_options', array());
        $this->auto_posting_enabled = isset($this->options['auto_posting_enabled']) ? 
            (bool)$this->options['auto_posting_enabled'] : true;
        
        $this->init_hooks();
        
        // Инициализируем админ-панель только в админке
        if (is_admin()) {
            new OkAutoPostAdmin($this);
        }
    }
    
    /**
     * Проверка, является ли версия PRO
     */
    public function is_pro_version() {
        return OkAutoPostLicensing::is_license_valid();
    }
    
    private function init_hooks() {
        add_action('init', array($this, 'init'));
        
        // Регистрируем хуки только если автопостинг включен
        if ($this->auto_posting_enabled && !self::$hooks_registered) {
            $this->register_auto_posting_hooks();
        }
        
        // Регистрируем хук для отложенной публикации
        add_action('ok_delayed_auto_post', array($this, 'delayed_auto_post_to_ok'), 10, 1);
    }
    
    /**
     * Регистрация хуков автопостинга
     */
    private function register_auto_posting_hooks() {
        if (self::$hooks_registered) {
            return;
        }
        
        if (has_action('transition_post_status', array($this, 'on_post_status_change'))) {
            self::$hooks_registered = true;
            return;
        }
        
        add_action('transition_post_status', array($this, 'on_post_status_change'), 10, 3);
        add_action('post_updated', array($this, 'on_post_date_change'), 10, 3);
        add_action('publish_future_post', array($this, 'on_future_post_published'), 10, 1);
        
        self::$hooks_registered = true;
    }
    
    /**
     * Удаление хуков автопостинга
     */
    private function unregister_auto_posting_hooks() {
        if (!self::$hooks_registered) {
            return;
        }
        
        remove_action('transition_post_status', array($this, 'on_post_status_change'), 10);
        remove_action('post_updated', array($this, 'on_post_date_change'), 10);
        remove_action('publish_future_post', array($this, 'on_future_post_published'), 10);
        
        self::$hooks_registered = false;
    }
    
    public function init() {
        load_plugin_textdomain('ok-auto-post', false, dirname(plugin_basename(__FILE__)) . '/languages');
    }
    
    /**
     * Обработка изменения статуса поста
     */
    public function on_post_status_change($new_status, $old_status, $post) {
        if ($new_status !== 'publish' || $old_status === 'publish') {
            return;
        }
        
        if (!is_object($post) || !isset($post->ID)) {
            return;
        }
        
        if (wp_is_post_autosave($post->ID) || wp_is_post_revision($post->ID)) {
            return;
        }
        
        $allowed_types = $this->options['post_types'] ?? array('post');
        if (!in_array($post->post_type, $allowed_types)) {
            return;
        }
        
        $active_groups = $this->options['group_ids_array'] ?? array();
        if (empty($active_groups)) {
            OkAutoPostLogger::log_debug('No active groups for post ' . $post->ID);
            return;
        }
        
        wp_schedule_single_event(time() + 10, 'ok_delayed_auto_post', array($post->ID));
        OkAutoPostLogger::log_debug('Scheduled delayed post for ID: ' . $post->ID);
    }
    
    /**
     * Обработка изменения даты публикации поста
     */
    public function on_post_date_change($post_id, $post_after, $post_before) {
        if (!is_object($post_after) || !is_object($post_before)) {
            return;
        }
        
        if ($post_after->post_status !== 'publish') {
            return;
        }
        
        if ($post_after->post_date === $post_before->post_date) {
            return;
        }
        
        if (wp_is_post_autosave($post_id) || wp_is_post_revision($post_id)) {
            return;
        }
        
        $allowed_types = $this->options['post_types'] ?? array('post');
        if (!in_array($post_after->post_type, $allowed_types)) {
            return;
        }
        
        $post_date = strtotime($post_after->post_date_gmt);
        $current_time = current_time('timestamp', true);
        
        if ($post_date > $current_time) {
            OkAutoPostLogger::log_debug('Post ' . $post_id . ' scheduled for future publication: ' . $post_after->post_date);
            
            delete_post_meta($post_id, '_ok_auto_posted');
            wp_schedule_single_event($post_date, 'ok_delayed_auto_post', array($post_id));
            OkAutoPostLogger::log_debug('Scheduled OK post for future date: ' . $post_after->post_date);
        } else {
            delete_post_meta($post_id, '_ok_auto_posted');
            wp_schedule_single_event(time() + 10, 'ok_delayed_auto_post', array($post_id));
            OkAutoPostLogger::log_debug('Post date changed to past, scheduling immediate post: ' . $post_id);
        }
    }
    
    /**
     * Обработка публикации запланированного поста
     */
    public function on_future_post_published($post_id) {
        $post = get_post($post_id);
        
        if (!$post || $post->post_status !== 'publish') {
            return;
        }
        
        if (wp_is_post_autosave($post_id) || wp_is_post_revision($post_id)) {
            return;
        }
        
        $allowed_types = $this->options['post_types'] ?? array('post');
        if (!in_array($post->post_type, $allowed_types)) {
            return;
        }
        
        $active_groups = $this->options['group_ids_array'] ?? array();
        if (empty($active_groups)) {
            OkAutoPostLogger::log_debug('No active groups for future post ' . $post_id);
            return;
        }
        
        wp_schedule_single_event(time() + 10, 'ok_delayed_auto_post', array($post_id));
        OkAutoPostLogger::log_debug('Scheduled delayed post for future post ID: ' . $post_id);
    }
    
    /**
     * Отложенная автоматическая публикация
     */
    public function delayed_auto_post_to_ok($post_id) {
        $post = get_post($post_id);
        
        if (!$post) {
            OkAutoPostLogger::log_debug('Post ' . $post_id . ' not found');
            return;
        }
        
        if ($post->post_status !== 'publish') {
            OkAutoPostLogger::log_debug('Post ' . $post_id . ' is not published (status: ' . $post->post_status . ')');
            return;
        }
        
        $post_date = strtotime($post->post_date_gmt);
        $current_time = current_time('timestamp', true);
        
        if ($post_date > $current_time) {
            OkAutoPostLogger::log_debug('Post ' . $post_id . ' is scheduled for future, rescheduling...');
            wp_schedule_single_event($post_date, 'ok_delayed_auto_post', array($post_id));
            return;
        }
        
        if (get_post_meta($post_id, '_ok_auto_posted', true)) {
            OkAutoPostLogger::log_debug('Post ' . $post_id . ' already posted to OK');
            return;
        }
        
        $lock_key = 'ok_posting_lock_' . $post_id;
        if (get_transient($lock_key)) {
            OkAutoPostLogger::log_debug('Post ' . $post_id . ' is already being processed');
            return;
        }
        
        set_transient($lock_key, true, 60);
        
        OkAutoPostLogger::log_debug('Starting delayed OK auto-post for post ID: ' . $post_id);
        
        try {
            $active_groups = $this->options['group_ids_array'] ?? array();
            
            if (empty($active_groups)) {
                throw new Exception('No active groups configured');
            }
            
            $api = new OkAutoPostAPI($this->options);
            
            // Используем новый метод из класса API для подготовки сообщения
            // который использует стрелку » вместо &hellip;
            $message = $api->prepare_message_from_template($post);
            
            $api->post_to_groups($active_groups, $message, $post);
            
            update_post_meta($post_id, '_ok_auto_posted', current_time('mysql'));
            OkAutoPostLogger::log_debug('Successfully posted to OK and marked as posted: ' . $post_id);
            
        } catch (Exception $e) {
            OkAutoPostLogger::log_debug('Error posting to OK for post ' . $post_id . ': ' . $e->getMessage());
        } finally {
            delete_transient($lock_key);
        }
    }
    
    /**
     * Подготовить сообщение для поста (старый метод - теперь не используется)
     */
    private function prepare_post_message($post) {
        $template = $this->options['post_template'] ?? '{title}

{excerpt}

Читать далее: {url}';
        
        $author = get_the_author_meta('display_name', $post->post_author);
        
        $clean_content = wp_strip_all_tags(strip_shortcodes($post->post_content));
        $clean_content = preg_replace('/<!--\s*wp:.*?-->/', '', $clean_content);
        $clean_content = trim($clean_content);
        
        $excerpt = $post->post_excerpt ?: wp_trim_words($clean_content, 55);
        $site_name = get_bloginfo('name');
        
        $replacements = array(
            '{title}' => get_the_title($post),
            '{excerpt}' => $excerpt,
            '{content}' => $clean_content,
            '{url}' => get_permalink($post),
            '{author}' => $author,
            '{site_name}' => $site_name
        );
        
        return str_replace(array_keys($replacements), array_values($replacements), $template);
    }
    
    /**
     * Переключение автопостинга
     */
    public function toggle_auto_posting($enable) {
        $this->options = get_option('ok_auto_post_options', array());
        $this->options['auto_posting_enabled'] = $enable;
        update_option('ok_auto_post_options', $this->options);
        
        $this->auto_posting_enabled = $enable;
        
        if ($enable) {
            $this->register_auto_posting_hooks();
        } else {
            $this->unregister_auto_posting_hooks();
        }
        
        return true;
    }
    
    /**
     * Получить состояние автопостинга
     */
    public function is_auto_posting_enabled() {
        return $this->auto_posting_enabled;
    }
    
    /**
     * Получить настройки
     */
    public function get_options() {
        return $this->options;
    }
    
    /**
     * Очистка всех данных плагина
     */
    public function cleanup_plugin_data() {
        delete_option('ok_auto_post_options');
        delete_option('ok_auto_post_logs');
        delete_option('ok_auto_post_debug_logs');
        
        $this->cleanup_post_meta();
        $this->cleanup_cron_jobs();
    }
    
    /**
     * Очистка мета-данных постов
     */
    private function cleanup_post_meta() {
        global $wpdb;
        
        $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM {$wpdb->postmeta} WHERE meta_key = %s",
                '_ok_auto_posted'
            )
        );
        
        $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM {$wpdb->options} WHERE option_name LIKE %s",
                '_transient_ok_posting_lock_%'
            )
        );
    }
    
    /**
     * Очистка крон-задач
     */
    private function cleanup_cron_jobs() {
        wp_clear_scheduled_hook('ok_delayed_auto_post');
    }
}