<?php

class OkAutoPostAPI {
    
    private $options;
    private $api_url = 'https://api.ok.ru/fb.do';
    
    public function __construct($options) {
        $this->options = $options;
    }
    
    /**
     * Проверка PRO версии
     */
    private function is_pro_version() {
        return !empty($this->options['license_key']) && 
               isset($this->options['license_status']) && 
               $this->options['license_status'] === 'valid';
    }
    
    /**
     * Тестирование подключения к API
     */
    public function test_api_connection($credentials = null) {
        if ($credentials === null) {
            $credentials = $this->options;
        }
        
        $method = 'users.getCurrentUser';
        $params = array(
            'fields' => 'uid,name,first_name,last_name'
        );
        
        try {
            $response = $this->make_api_request($method, $params, $credentials);
            
            if (!isset($response['uid'])) {
                throw new Exception('Неверный ответ от API');
            }
            
            return $response;
        } catch (Exception $e) {
            throw new Exception($e->getMessage());
        }
    }
    
    /**
     * Получить эксцерпт для Одноклассников со стрелкой вместо &hellip;
     */
    private function get_ok_excerpt($post) {
        // Временный фильтр только для OK
        add_filter('excerpt_more', array($this, 'ok_excerpt_more'));
        $excerpt = get_the_excerpt($post);
        remove_filter('excerpt_more', array($this, 'ok_excerpt_more'));
        
        return $excerpt;
    }
    
    /**
     * Кастомное "многоточие" для Одноклассников
     */
    public function ok_excerpt_more($more) {
        return ' »'; // Стрелка вместо &hellip;
    }
    
    /**
     * Подготовить сообщение по шаблону с учетом версии
     */
    public function prepare_message_from_template($post, $template = null) {
        if ($template === null) {
            $template = $this->options['post_template'] ?? '{excerpt}

Читать далее: {url}';
        }
        
        // Используем наш метод с кастомным "многоточием"
        $excerpt = $this->get_ok_excerpt($post);
        
        // Базовые замены для FREE версии
        $replacements = array(
            '{excerpt}' => $excerpt,
            '{url}' => get_permalink($post->ID),
        );
        
        // PRO переменные - только для PRO версии
        if ($this->is_pro_version()) {
            $pro_replacements = array(
                '{title}' => get_the_title($post),
                '{content}' => wp_strip_all_tags(strip_shortcodes($post->post_content)),
                '{author}' => get_the_author_meta('display_name', $post->post_author),
                '{site_name}' => get_bloginfo('name'),
            );
            $replacements = array_merge($replacements, $pro_replacements);
        } else {
            // В FREE версии заменяем PRO переменные на пустые строки
            $pro_replacements = array(
                '{title}' => '',
                '{content}' => '',
                '{author}' => '',
                '{site_name}' => '',
            );
            $replacements = array_merge($replacements, $pro_replacements);
        }
        
        $message = str_replace(array_keys($replacements), array_values($replacements), $template);
        
        // Удаляем пустые строки, которые могли образоваться из-за удаленных переменных
        $message = preg_replace('/\n\s*\n\s*\n/', "\n\n", $message);
        $message = trim($message);
        
        // Если после удаления переменных осталась только ссылка, добавляем базовый текст
        if (empty(trim(str_replace(array($excerpt, get_permalink($post->ID)), '', $message)))) {
            $message = "Новый пост с сайта\n\n" . $message;
        }
        
        return $message;
    }
    
    /**
     * Опубликовать в группы
     */
    public function post_to_groups($group_ids, $message, $post = null) {
        OkAutoPostLogger::log_debug('Posting to ' . count($group_ids) . ' groups for post ID: ' . ($post ? $post->ID : 'test'));
        
        $success_count = 0;
        $error_count = 0;
        
        foreach ($group_ids as $group_id) {
            try {
                OkAutoPostLogger::log_debug('Attempting to post to group: ' . $group_id);
                $result = $this->post_to_group($group_id, $message, $post);
                
                OkAutoPostLogger::log_post($post ? $post->ID : 0, $group_id, 'success', 'Пост опубликован');
                OkAutoPostLogger::log_debug('Successfully posted to group: ' . $group_id);
                $success_count++;
                
            } catch (Exception $e) {
                OkAutoPostLogger::log_post($post ? $post->ID : 0, $group_id, 'error', $e->getMessage());
                OkAutoPostLogger::log_debug('Error posting to group ' . $group_id . ': ' . $e->getMessage());
                $error_count++;
            }
        }
        
        OkAutoPostLogger::log_debug('Posting completed: ' . $success_count . ' success, ' . $error_count . ' errors');
    }
    
    /**
     * Опубликовать в конкретную группу
     */
    public function post_to_group($group_id, $message, $post = null) {
        $method = 'mediatopic.post';
        
        $media = array();
        
        if (!empty($message)) {
            $media[] = array(
                'type' => 'text',
                'text' => $message
            );
        }
        
        $image_url = '';
        if ($post !== null) {
            $image_url = $this->get_post_image($post);
        }
        
        if (!empty($image_url)) {
            try {
                $photo_token = $this->upload_photo_to_ok($image_url, $group_id);
                
                $media[] = array(
                    'type' => 'photo',
                    'list' => array(
                        array(
                            'id' => $photo_token
                        )
                    )
                );
                
                OkAutoPostLogger::log_debug('Image uploaded successfully with token: ' . $photo_token);
                
            } catch (Exception $e) {
                OkAutoPostLogger::log_debug('Image upload failed: ' . $e->getMessage());
            }
        }
        
        if (empty($media)) {
            $media[] = array(
                'type' => 'text',
                'text' => 'Новый пост с сайта ' . get_bloginfo('name')
            );
        }
        
        $attachment = array(
            'media' => $media
        );
        
        $params = array(
            'gid' => $group_id,
            'type' => 'GROUP_THEME',
            'attachment' => json_encode($attachment, JSON_UNESCAPED_UNICODE)
        );
        
        OkAutoPostLogger::log_debug('Posting to group ' . $group_id . ' with media: ' . count($media) . ' items');
        
        return $this->make_api_request($method, $params);
    }
    
    /**
     * Получить изображение поста
     */
    private function get_post_image($post) {
        if (has_post_thumbnail($post->ID)) {
            $image_url = get_the_post_thumbnail_url($post->ID, 'large');
            if ($image_url) {
                OkAutoPostLogger::log_debug('Featured image found: ' . $image_url);
                return $image_url;
            }
        }
        
        preg_match('/<img.+src=[\'"]([^\'"]+)[\'"].*>/i', $post->post_content, $matches);
        if (!empty($matches[1])) {
            OkAutoPostLogger::log_debug('Content image found: ' . $matches[1]);
            return $matches[1];
        }
        
        OkAutoPostLogger::log_debug('No image found for post ' . $post->ID);
        return '';
    }
    
    /**
     * Загрузить изображение в Одноклассники
     */
    private function upload_photo_to_ok($image_url, $group_id) {
        $method = 'photosV2.getUploadUrl';
        $params = array(
            'gid' => $group_id
        );
        
        $upload_data = $this->make_api_request($method, $params);
        
        if (!isset($upload_data['upload_url'])) {
            throw new Exception('Не удалось получить URL для загрузки фото');
        }
        
        $upload_url = $upload_data['upload_url'];
        
        $image_response = wp_remote_get($image_url);
        if (is_wp_error($image_response)) {
            throw new Exception('Не удалось скачать изображение: ' . $image_response->get_error_message());
        }
        
        $image_data = wp_remote_retrieve_body($image_response);
        
        $upload_response = wp_remote_post($upload_url, array(
            'body' => $image_data,
            'headers' => array(
                'Content-Type' => 'application/octet-stream'
            ),
            'timeout' => 30
        ));
        
        if (is_wp_error($upload_response)) {
            throw new Exception('Ошибка загрузки фото: ' . $upload_response->get_error_message());
        }
        
        $upload_result = json_decode(wp_remote_retrieve_body($upload_response), true);
        
        if (!isset($upload_result['photos'])) {
            throw new Exception('Неверный ответ при загрузке фото');
        }
        
        $photo = reset($upload_result['photos']);
        return $photo['token'];
    }
    
    /**
     * Базовый метод для API запросов
     */
    public function make_api_request($method, $params = array(), $credentials = null) {
        if ($credentials === null) {
            $credentials = $this->options;
        }
        
        $base_params = array(
            'application_key' => $credentials['app_key'],
            'method' => $method,
            'format' => 'json'
        );
        
        $params = array_merge($base_params, $params);
        
        ksort($params);
        
        $signature = '';
        foreach ($params as $key => $value) {
            $signature .= $key . '=' . $value;
        }
        
        $secret_key = $credentials['app_secret_key'] ?? '';
        $signature .= md5($credentials['access_token'] . $secret_key);
        $params['sig'] = md5($signature);
        $params['access_token'] = $credentials['access_token'];
        
        $url = $this->api_url . '?' . http_build_query($params);
        
        OkAutoPostLogger::log_debug('API Request: ' . $url);
        
        $response = wp_remote_get($url, array(
            'timeout' => 30,
            'sslverify' => false
        ));
        
        if (is_wp_error($response)) {
            $error_message = 'Ошибка HTTP: ' . $response->get_error_message();
            OkAutoPostLogger::log_debug($error_message);
            throw new Exception($error_message);
        }
        
        $body = wp_remote_retrieve_body($response);
        OkAutoPostLogger::log_debug('API Response: ' . $body);
        
        $data = json_decode($body, true);
        
        if (isset($data['error_code'])) {
            $error_message = 'Ошибка API: ' . $data['error_msg'] . ' (код: ' . $data['error_code'] . ')';
            OkAutoPostLogger::log_debug($error_message);
            throw new Exception($error_message);
        }
        
        return $data;
    }
    
    /**
     * Подготовить тестовое сообщение
     */
    public function prepare_test_message($template = null) {
        if ($template === null) {
            $template = $this->options['post_template'] ?? '{excerpt}

Читать далее: {url}';
        }
        
        // Создаем тестовый объект поста
        $test_post = (object) array(
            'ID' => 0,
            'post_title' => 'Тестовый пост от ' . get_bloginfo('name'),
            'post_content' => 'Это тестовое сообщение для проверки работы плагина авто-постинга в Одноклассники.',
            'post_excerpt' => 'Это тестовое сообщение для проверки работы плагина авто-постинга в Одноклассниках.',
            'post_author' => 1
        );
        
        // Используем стандартный метод для согласованности
        return $this->prepare_message_from_template($test_post, $template);
    }
}