<?php

class OkAutoPostAdmin {
    
    private $core;
    private $options;
    private $current_tab = 'settings';
    private $admin_notices = array();
    
    public function __construct($core) {
        $this->core = $core;
        $this->options = $core->get_options();
        $this->current_tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : 'settings';
        
        $this->init_hooks();
        $this->handle_license_actions();
    }
    
    private function init_hooks() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'admin_init'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
        add_action('admin_notices', array($this, 'show_admin_notices'));
        
        // Уведомление об обновлении плагина
        add_action('admin_notices', array($this, 'show_update_notice'));
        
        // AJAX обработчики
        add_action('wp_ajax_ok_test_connection', array($this, 'ajax_test_connection'));
        add_action('wp_ajax_ok_test_post', array($this, 'ajax_test_post'));
        add_action('wp_ajax_ok_clear_debug_logs', array($this, 'ajax_clear_debug_logs'));
        add_action('wp_ajax_ok_clear_post_logs', array($this, 'ajax_clear_post_logs'));
        add_action('wp_ajax_ok_cleanup_all_data', array($this, 'ajax_cleanup_all_data'));
        add_action('wp_ajax_ok_republish_post', array($this, 'ajax_republish_post'));
        add_action('wp_ajax_ok_clear_post_meta', array($this, 'ajax_clear_post_meta'));
        add_action('wp_ajax_ok_clear_all_meta', array($this, 'ajax_clear_all_meta'));
        add_action('wp_ajax_ok_toggle_auto_posting', array($this, 'ajax_toggle_auto_posting'));
    }
    
    /**
     * Обработка действий с лицензией
     */
    private function handle_license_actions() {
        if (!isset($_POST['action']) || !isset($_POST['ok_license_nonce'])) {
            return;
        }
        
        if (!wp_verify_nonce($_POST['ok_license_nonce'], 'ok_activate_license') && 
            !wp_verify_nonce($_POST['ok_license_nonce'], 'ok_deactivate_license')) {
            $this->add_admin_notice('Неверный nonce-токен', 'error');
            return;
        }
        
        if (!current_user_can('manage_options')) {
            $this->add_admin_notice('Недостаточно прав', 'error');
            return;
        }
        
        $action = sanitize_text_field($_POST['action']);
        
        if ($action === 'activate_license') {
            $this->activate_license();
        } elseif ($action === 'deactivate_license') {
            $this->deactivate_license();
        }
    }
    
    /**
     * Добавление административного уведомления
     */
    private function add_admin_notice($message, $type = 'success') {
        $this->admin_notices[] = array(
            'message' => $message,
            'type' => $type
        );
    }
    
    /**
     * Активация лицензии
     */
    private function activate_license() {
        $license_key = sanitize_text_field($_POST['license_key'] ?? '');
        $license_email = sanitize_email($_POST['license_email'] ?? '');
        
        if (empty($license_key) || empty($license_email)) {
            $this->add_admin_notice('Пожалуйста, заполните оба поля: лицензионный ключ и email.', 'error');
            return;
        }
        
        // Проверка формата ключа
        if (!OkAutoPostLicensing::validate_license_format($license_key)) {
            $this->add_admin_notice('Неверный формат лицензионного ключа. Ожидается формат: OK-PRO-XXXX-XXXX-XXXX-XXXX-XXXX', 'error');
            return;
        }
        
        try {
            // Активация на сервере лицензий
            $license_data = OkAutoPostLicensing::activate_on_server($license_key, $license_email);
            
            // Сохраняем данные лицензии
            OkAutoPostLicensing::save_license_key($license_key, $license_email);
            
            // Обновляем статус лицензии
            update_option(OKAP_LICENSE_STATUS_OPTION, 'valid');
            update_option(OKAP_LICENSE_DATA_OPTION, $license_data);
            update_option(OKAP_LICENSE_LAST_CHECK_OPTION, time());
            delete_option(OKAP_LICENSE_ERROR_OPTION);
            
            // Обновляем опции плагина
            $this->options['license_key'] = $license_key;
            $this->options['license_email'] = $license_email;
            $this->options['license_status'] = 'valid';
            $this->options['license_expires'] = $license_data['expires_at'] ?? date('Y-m-d', strtotime('+1 year'));
            update_option('ok_auto_post_options', $this->options);
            
            $this->add_admin_notice('✅ PRO версия успешно активирована! Теперь доступны все функции.');
            
        } catch (Exception $e) {
            update_option(OKAP_LICENSE_ERROR_OPTION, $e->getMessage());
            $this->add_admin_notice('Ошибка активации лицензии: ' . $e->getMessage(), 'error');
        }
    }
    
    /**
     * Деактивация лицензии
     */
    private function deactivate_license() {
        $license_key = get_option(OKAP_LICENSE_KEY_OPTION);
        $site_url = home_url();
        
        if (empty($license_key)) {
            $this->add_admin_notice('Лицензионный ключ не найден.', 'error');
            return;
        }
        
        try {
            // Деактивация на сервере
            OkAutoPostLicensing::deactivate_on_server($license_key, $site_url);
            
            // Удаляем локальные данные
            OkAutoPostLicensing::remove_license_key();
            
            // Обновляем опции плагина
            unset($this->options['license_key']);
            unset($this->options['license_email']);
            unset($this->options['license_status']);
            unset($this->options['license_expires']);
            update_option('ok_auto_post_options', $this->options);
            
            $this->add_admin_notice('✅ Лицензия деактивирована. Вы перешли на FREE версию.');
            
        } catch (Exception $e) {
            update_option(OKAP_LICENSE_ERROR_OPTION, $e->getMessage());
            $this->add_admin_notice('Ошибка деактивации: ' . $e->getMessage(), 'error');
        }
    }
    
    /**
     * Показывает уведомление об обновлении на страницах плагина
     */
    public function show_update_notice() {
        // Показываем только на страницах нашего плагина
        $screen = get_current_screen();
        if (!$screen || strpos($screen->id, 'ok-auto-post') === false) {
            return;
        }
        
        $update_plugins = get_site_transient('update_plugins');
        
        if (isset($update_plugins->response[OK_AUTO_POST_PLUGIN_BASENAME])) {
            $update_data = $update_plugins->response[OK_AUTO_POST_PLUGIN_BASENAME];
            echo '<div class="notice notice-warning is-dismissible">';
            echo '<p>🚀 <strong>Доступна новая версия OK Auto Post (' . esc_html($update_data->new_version) . ')</strong>. ';
            echo '<a href="' . admin_url('update-core.php') . '" class="button button-primary" style="margin-left: 10px;">Обновить сейчас</a>';
            echo '</p></div>';
        }
        
        // Показываем уведомление о проверке лицензии
        $license_status = get_option(OKAP_LICENSE_STATUS_OPTION);
        $license_error = get_option(OKAP_LICENSE_ERROR_OPTION);
        
        if ($license_status === 'invalid' && !empty($license_error)) {
            echo '<div class="notice notice-error is-dismissible">';
            echo '<p>⚠️ <strong>Проблема с лицензией:</strong> ' . esc_html($license_error) . '</p>';
            echo '<p>Проверьте лицензию на <a href="' . admin_url('options-general.php?page=ok-auto-post&tab=license') . '">вкладке "Лицензия PRO"</a></p>';
            echo '</div>';
        } elseif ($license_status === 'expired') {
            echo '<div class="notice notice-warning is-dismissible">';
            echo '<p>⚠️ <strong>Срок действия лицензии истек!</strong> Обновите лицензию для продолжения работы PRO версии.</p>';
            echo '<p><a href="https://gaagaa.ru/shop/wpplugins/OK_Auto_Post/ok-auto-post.html" target="_blank" class="button button-primary">Обновить лицензию</a></p>';
            echo '</div>';
        }
    }
    
    /**
     * Подключение CSS и JavaScript для админ-панели
     */
    public function enqueue_admin_assets($hook) {
        if ($hook !== 'settings_page_ok-auto-post') {
            return;
        }
        
        // CSS
        $css_path = OK_AUTO_POST_PLUGIN_PATH . 'assets/css/admin.css';
        if (file_exists($css_path)) {
            wp_enqueue_style(
                'ok-publisher-admin',
                OK_AUTO_POST_PLUGIN_URL . 'assets/css/admin.css',
                array(),
                OK_AUTO_POST_VERSION
            );
        } else {
            // Базовый CSS если файл не найден
            wp_enqueue_style(
                'ok-publisher-admin-inline',
                OK_AUTO_POST_PLUGIN_URL . 'assets/css/admin.css',
                array(),
                OK_AUTO_POST_VERSION
            );
        }
        
        // JavaScript
        $js_path = OK_AUTO_POST_PLUGIN_PATH . 'assets/js/admin.js';
        if (file_exists($js_path)) {
            wp_enqueue_script(
                'ok-publisher-admin',
                OK_AUTO_POST_PLUGIN_URL . 'assets/js/admin.js',
                array('jquery'),
                OK_AUTO_POST_VERSION,
                true
            );
        } else {
            // Базовый JS если файл не найден
            wp_enqueue_script(
                'ok-publisher-admin-inline',
                OK_AUTO_POST_PLUGIN_URL . 'assets/js/admin.js',
                array('jquery'),
                OK_AUTO_POST_VERSION,
                true
            );
        }
        
        // Локализация для JS
        wp_localize_script('ok-publisher-admin', 'ok_auto_post', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonces' => array(
                'test_connection' => wp_create_nonce('ok_test_connection'),
                'test_post' => wp_create_nonce('ok_test_post'),
                'toggle_auto_posting' => wp_create_nonce('ok_toggle_auto_posting'),
                'clear_post_meta' => wp_create_nonce('ok_clear_post_meta'),
                'clear_post_logs' => wp_create_nonce('ok_clear_post_logs'),
                'clear_debug_logs' => wp_create_nonce('ok_clear_debug_logs'),
                'clear_all_meta' => wp_create_nonce('ok_clear_all_meta'),
                'cleanup_all_data' => wp_create_nonce('ok_cleanup_all_data')
            )
        ));
    }
    
    /**
     * Добавление меню в админ-панель
     */
    public function add_admin_menu() {
        add_options_page(
            'Auto Post to OK',
            'OK Auto Post',
            'manage_options',
            'ok-auto-post',
            array($this, 'settings_page')
        );
    }
    
    /**
     * Инициализация настроек
     */
    public function admin_init() {
        register_setting('ok_auto_post_settings', 'ok_auto_post_options', array($this, 'validate_options'));
        
        add_settings_section(
            'ok_api_section',
            'Данные приложения Одноклассников',
            array($this, 'api_section_callback'),
            'ok-auto-post'
        );
        
        add_settings_field(
            'auto_posting_control',
            'Управление автопостингом',
            array($this, 'auto_posting_control_callback'),
            'ok-auto-post',
            'ok_api_section'
        );
        
        add_settings_field(
            'app_id',
            'Application ID',
            array($this, 'app_id_callback'),
            'ok-auto-post',
            'ok_api_section'
        );
        
        add_settings_field(
            'app_key',
            'Публичный ключ приложения',
            array($this, 'app_key_callback'),
            'ok-auto-post',
            'ok_api_section'
        );
        
        add_settings_field(
            'app_secret_key',
            'Session_secret_key',
            array($this, 'app_secret_key_callback'),
            'ok-auto-post',
            'ok_api_section'
        );
        
        add_settings_field(
            'access_token',
            'Вечный access_token',
            array($this, 'access_token_callback'),
            'ok-auto-post',
            'ok_api_section'
        );
        
        add_settings_field(
            'test_connection',
            'Проверка подключения',
            array($this, 'test_connection_callback'),
            'ok-auto-post',
            'ok_api_section'
        );
        
        add_settings_section(
            'ok_groups_section',
            'Настройки групп для постинга',
            array($this, 'groups_section_callback'),
            'ok-auto-post'
        );
        
        add_settings_field(
            'group_ids',
            'ID групп для постинга',
            array($this, 'group_ids_callback'),
            'ok-auto-post',
            'ok_groups_section'
        );
        
        add_settings_field(
            'test_post',
            'Тестовая публикация',
            array($this, 'test_post_callback'),
            'ok-auto-post',
            'ok_groups_section'
        );
        
        add_settings_section(
            'ok_post_section',
            'Настройки постинга',
            array($this, 'post_section_callback'),
            'ok-auto-post'
        );
        
        add_settings_field(
            'post_types',
            'Типы записей для авто-постинга',
            array($this, 'post_types_callback'),
            'ok-auto-post',
            'ok_post_section'
        );
        
        add_settings_field(
            'post_template',
            'Шаблон поста',
            array($this, 'post_template_callback'),
            'ok-auto-post',
            'ok_post_section'
        );
    }
    
    /**
     * Callback функции для настроек
     */
    public function auto_posting_control_callback() {
        $auto_posting_enabled = $this->core->is_auto_posting_enabled();
        
        echo '<div style="margin-bottom: 15px;">';
        echo '<span id="ok-auto-posting-control-status" class="' . ($auto_posting_enabled ? 'status-enabled' : 'status-disabled') . '" style="font-weight: bold; margin-right: 10px;">';
        echo $auto_posting_enabled ? '✅ Автопостинг включен' : '⏸️ Автопостинг отключен';
        echo '</span>';
        echo '<button type="button" id="ok-auto-posting-control-btn" class="button ' . ($auto_posting_enabled ? 'button-warning' : 'button-primary') . '">';
        echo $auto_posting_enabled ? 'Выключить автопостинг' : 'Включить автопостинг';
        echo '</button>';
        echo '<span id="ok-auto-posting-control-result" style="margin-left: 10px;"></span>';
        echo '</div>';
        echo '<p class="description">Включите или выключите автоматическую публикацию новых записей в Одноклассниках</p>';
        
        echo '<input type="hidden" name="ok_auto_post_options[auto_posting_enabled]" value="' . ($auto_posting_enabled ? '1' : '0') . '" id="ok-auto-posting-control-field">';
    }
    
    public function api_section_callback() {
        echo '<div class="ok-instructions">';
        echo '<h3>Инструкция по настройке:</h3>';
        echo '<ol>';
        echo '<li><strong><a href="https://apiok.ru/dev/app/create" target="_blank">Создайте приложение</a></strong> в Одноклассниках</li>';
        echo '<li><strong>Заполните данные приложения</strong> ниже (можно найти в настройках приложения):';
        echo '<ul>';
        echo '<li><strong>Application ID</strong> - идентификатор приложения</li>';
        echo '<li><strong>Публичный ключ приложения</strong> (Application Key)</li>';
        echo '<li><strong>Секретный ключ</strong> (Session_secret_key) - <strong style="color: #d63638;">из OAuth-ответа (второй параметр)</strong></li>';
        echo '<li><strong>Вечный access_token</strong> - долгосрочный токен доступа - <strong style="color: #d63638;">из OAuth-ответа (первый параметр)</strong></li>';
        echo '</ul>';
        echo '</li>';
        echo '<li><strong>Укажите ID групп</strong> в которые нужно публиковать записи</li>';
        echo '<li><strong>Протестируйте отправку</strong> с помощью тестовой публикации</li>';
        echo '</ol>';
        echo '</div>';
    }
    
    public function app_id_callback() {
        $value = $this->options['app_id'] ?? '';
        echo '<input type="text" name="ok_auto_post_options[app_id]" value="' . esc_attr($value) . '" class="regular-text">';
        echo '<p class="description">Application ID (идентификатор приложения)</p>';
    }
    
    public function app_key_callback() {
        $value = $this->options['app_key'] ?? '';
        echo '<input type="text" name="ok_auto_post_options[app_key]" value="' . esc_attr($value) . '" class="regular-text">';
        echo '<p class="description">Публичный ключ приложения (Application Key)</p>';
    }
    
    public function app_secret_key_callback() {
        $value = $this->options['app_secret_key'] ?? '';
        echo '<input type="text" name="ok_auto_post_options[app_secret_key]" value="' . esc_attr($value) . '" class="regular-text">';
        echo '<p class="description">Секретный ключ (Session_secret_key) <strong style="color: #d63638;">Из OAuth-ответа (второй параметр)</strong></p>';
        
        echo '<div class="token-help" style="background: #f0f7ff; padding: 12px; margin: 8px 0; border-left: 4px solid #0073aa;">';
        echo '<p><strong>Где взять Session_secret_key:</strong></p>';
        echo '<ol style="margin-left: 20px; margin-bottom: 0;">';
        echo '<li>При использовании OAuth-авторизации вы получите ответ в формате:';
        echo '<pre style="background: #f7f7f7; padding: 8px; margin: 5px 0;">access_token=YOUR_ACCESS_TOKEN&session_secret_key=YOUR_SESSION_SECRET_KEY&expires_in=...</pre>';
        echo '</li>';
        echo '<li><strong>Session_secret_key</strong> - это второй параметр в ответе (<code>session_secret_key=...</code>)</li>';
        echo '<li>Он используется для подписи запросов к API Одноклассников</li>';
        echo '<li>Действует столько же, сколько и access_token (вечный или долгосрочный)</li>';
        echo '</ol>';
        echo '</div>';
    }
    
    public function access_token_callback() {
        $value = $this->options['access_token'] ?? '';
        echo '<input type="text" name="ok_auto_post_options[access_token]" value="' . esc_attr($value) . '" class="large-text">';
        echo '<p class="description">Вечный access_token с правами: VALUABLE_ACCESS, GROUP_CONTENT, PHOTO_CONTENT <strong style="color: #d63638;">Из OAuth-ответа (первый параметр)</strong></p>';
        
        echo '<div class="token-help" style="background: #f0f7ff; padding: 12px; margin: 8px 0; border-left: 4px solid #0073aa;">';
        echo '<p><strong>Где взять вечный access_token:</strong></p>';
        echo '<ol style="margin-left: 20px; margin-bottom: 0;">';
        echo '<li>При OAuth-авторизации вы получаете ответ в формате:';
        echo '<pre style="background: #f7f7f7; padding: 8px; margin: 5px 0;">access_token=YOUR_ACCESS_TOKEN&session_secret_key=YOUR_SESSION_SECRET_KEY&expires_in=...</pre>';
        echo '</li>';
        echo '<li><strong>Access_token</strong> - это первый параметр в ответе (<code>access_token=...</code>)</li>';
        echo '<li>Для получения вечного токена используйте тип авторизации "Долгосрочный токен" или "Вечный токен"</li>';
        echo '<li>Укажите права: <code>VALUABLE_ACCESS</code>, <code>GROUP_CONTENT</code>, <code>PHOTO_CONTENT</code></li>';
        echo '<li>Скопируйте значение параметра <code>access_token</code> из OAuth-ответа</li>';
        echo '</ol>';
        
        echo '<p><strong>Пример OAuth-ответа:</strong></p>';
        echo '<div style="background: #f7f7f7; padding: 10px; border: 1px solid #ddd; font-family: monospace; font-size: 13px; margin: 10px 0;">';
        echo 'access_token=<span style="color: #d63638; font-weight: bold;">tkn1_CUTt8vKHJmAPFxH...</span>&<br>';
        echo 'session_secret_key=<span style="color: #00a32a; font-weight: bold;">s8hG7kKJh98nB76gH...</span>&<br>';
        echo 'expires_in=0';
        echo '</div>';
        
        echo '<table style="width: 100%; border-collapse: collapse; margin-top: 10px;">';
        echo '<tr>';
        echo '<td style="padding: 5px; border: 1px solid #ddd; background: #f0f0f0; width: 30%;"><strong>access_token</strong></td>';
        echo '<td style="padding: 5px; border: 1px solid #ddd;">Первый параметр OAuth-ответа → вставляется в поле "Вечный access_token"</td>';
        echo '</tr>';
        echo '<tr>';
        echo '<td style="padding: 5px; border: 1px solid #ddd; background: #f0f0f0;"><strong>session_secret_key</strong></td>';
        echo '<td style="padding: 5px; border: 1px solid #ddd;">Второй параметр OAuth-ответа → вставляется в поле "Session_secret_key"</td>';
        echo '</tr>';
        echo '</table>';
        echo '</div>';
    }
    
    public function test_connection_callback() {
        echo '<button type="button" id="ok-test-connection" class="button">Проверить подключение к API</button>';
        echo '<span id="ok-test-result" style="margin-left: 10px;"></span>';
        echo '<p class="description">Проверит корректность введенных данных и подключение к Одноклассникам</p>';
    }
    
    public function groups_section_callback() {
        echo '<p>Укажите ID групп через запятую. Пример: <code>123456789,987654321</code></p>';
        echo '<p><strong>Как найти ID группы:</strong></p>';
        echo '<ol>';
        echo '<li>Перейдите в нужную группу в Одноклассниках</li>';
        echo '<li>Посмотрите в адресной строке - ID группы это цифры в URL</li>';
        echo '<li>Пример URL: <code>https://ok.ru/group/<strong>123456789</strong></code></li>';
        echo '</ol>';
    }
    
    public function group_ids_callback() {
        $value = $this->options['group_ids'] ?? '';
        echo '<textarea name="ok_auto_post_options[group_ids]" rows="3" class="large-text" placeholder="123456789,987654321">' . esc_textarea($value) . '</textarea>';
        echo '<p class="description">Введите ID групп через запятую. Только цифры.</p>';
    }
    
    public function test_post_callback() {
        echo '<button type="button" id="ok-test-post" class="button button-primary">Отправить тестовый пост</button>';
        echo '<span id="ok-test-post-result" style="margin-left: 10px;"></span>';
        echo '<p class="description">Отправит тестовое сообщение во все указанные группы для проверки работы</p>';
    }
    
    public function post_section_callback() {
        echo '<p>Настройте параметры автоматического постинга</p>';
    }
    
    public function post_types_callback() {
        $post_types = get_post_types(array('public' => true), 'objects');
        $selected_types = $this->options['post_types'] ?? array('post');
        
        foreach ($post_types as $post_type) {
            if ($post_type->name == 'attachment') continue;
            
            echo '<label style="display: block; margin: 5px 0;">';
            echo '<input type="checkbox" name="ok_auto_post_options[post_types][]" value="' . esc_attr($post_type->name) . '" ' . 
                 checked(in_array($post_type->name, $selected_types), true, false) . '>';
            echo ' ' . esc_html($post_type->label);
            echo '</label>';
        }
    }
    
    public function post_template_callback() {
        $value = $this->options['post_template'] ?? '{excerpt}

Читать далее: {url}';
        echo '<textarea name="ok_auto_post_options[post_template]" rows="5" class="large-text">' . esc_textarea($value) . '</textarea>';
        echo '<p class="description">В FREE версии доступны только 2 переменные:</p>';
        
        echo '<div class="ok-variables-info" style="background: #f9f9f9; padding: 12px; margin: 8px 0; border: 1px solid #ddd; border-radius: 4px;">';
        echo '<table style="width: 100%; border-collapse: collapse;">';
        echo '<thead><tr style="background: #f0f0f0;"><th style="padding: 8px; text-align: left; border-bottom: 1px solid #ddd;">Переменная</th><th style="padding: 8px; text-align: left; border-bottom: 1px solid #ddd;">Описание</th><th style="padding: 8px; text-align: left; border-bottom: 1px solid #ddd;">Доступность</th></tr></thead>';
        echo '<tbody>';
        
        // Сначала FREE переменные
        echo '<tr><td style="padding: 8px; border-bottom: 1px solid #eee;"><code>{excerpt}</code></td><td style="padding: 8px; border-bottom: 1px solid #eee;">Краткое описание/анонс записи</td><td style="padding: 8px; border-bottom: 1px solid #eee;"><span style="color: #46b450; font-weight: bold;">FREE версия</span></td></tr>';
        echo '<tr><td style="padding: 8px; border-bottom: 1px solid #eee;"><code>{url}</code></td><td style="padding: 8px; border-bottom: 1px solid #eee;">Прямая ссылка на запись</td><td style="padding: 8px; border-bottom: 1px solid #eee;"><span style="color: #46b450; font-weight: bold;">FREE версия</span></td></tr>';
        
        // Разделитель
        echo '<tr style="background: #f9f9f9;"><td colspan="3" style="padding: 8px; border-bottom: 1px solid #eee;"><hr style="border: none; border-top: 1px dashed #ddd;"></td></tr>';
        
        // Затем PRO переменные
        $pro_disabled = !$this->core->is_pro_version();
        $pro_style = $pro_disabled ? 'style="opacity: 0.6;"' : '';
        
        echo '<tr ' . $pro_style . '><td style="padding: 8px; border-bottom: 1px solid #eee;"><code>{title}</code></td><td style="padding: 8px; border-bottom: 1px solid #eee;">Заголовок записи WordPress</td><td style="padding: 8px; border-bottom: 1px solid #eee;">';
        if ($pro_disabled) {
            echo '<span style="color: #dc3232; font-weight: bold;">PRO версия</span>';
        } else {
            echo '<span style="color: #46b450; font-weight: bold;">✅ Доступно</span>';
        }
        echo '</td></tr>';
        
        echo '<tr ' . $pro_style . '><td style="padding: 8px; border-bottom: 1px solid #eee;"><code>{content}</code></td><td style="padding: 8px; border-bottom: 1px solid #eee;">Полное содержимое записи</td><td style="padding: 8px; border-bottom: 1px solid #eee;">';
        if ($pro_disabled) {
            echo '<span style="color: #dc3232; font-weight: bold;">PRO версия</span>';
        } else {
            echo '<span style="color: #46b450; font-weight: bold;">✅ Доступно</span>';
        }
        echo '</td></tr>';
        
        echo '<tr ' . $pro_style . '><td style="padding: 8px; border-bottom: 1px solid #eee;"><code>{author}</code></td><td style="padding: 8px; border-bottom: 1px solid #eee;">Имя автора записи</td><td style="padding: 8px; border-bottom: 1px solid #eee;">';
        if ($pro_disabled) {
            echo '<span style="color: #dc3232; font-weight: bold;">PRO версия</span>';
        } else {
            echo '<span style="color: #46b450; font-weight: bold;">✅ Доступно</span>';
        }
        echo '</td></tr>';
        
        echo '<tr ' . $pro_style . '><td style="padding: 8px; border-bottom: 1px solid #eee;"><code>{site_name}</code></td><td style="padding: 8px; border-bottom: 1px solid #eee;">Название сайта WordPress</td><td style="padding: 8px; border-bottom: 1px solid #eee;">';
        if ($pro_disabled) {
            echo '<span style="color: #dc3232; font-weight: bold;">PRO версия</span>';
        } else {
            echo '<span style="color: #46b450; font-weight: bold;">✅ Доступно</span>';
        }
        echo '</td></tr>';
        
        echo '</tbody>';
        echo '</table>';
        echo '</div>';
        
        if (!$this->core->is_pro_version()) {
            // Информация о FREE версии
            echo '<div style="background: #f9f9f9; padding: 12px; margin: 12px 0; border: 1px solid #e0e0e0; border-radius: 4px;">';
            echo '<p style="margin: 0 0 8px 0; color: #555; display: flex; align-items: center;">';
            echo '<span style="margin-right: 8px; color: #46b450;">ℹ️</span> <strong>Вы используете FREE версию.</strong>';
            echo '</p>';
            echo '<p style="margin: 8px 0 0 0; color: #666; font-size: 13px;">';
            echo 'Доступны только 2 переменные: <code>{excerpt}</code> и <code>{url}</code>';
            echo '</p>';
            echo '</div>';
            
            // Призыв к апгрейду
            echo '<div style="background: #f0f7ff; padding: 12px; margin: 12px 0; border: 1px solid #c3d9ff; border-radius: 4px;">';
            echo '<p style="margin: 0 0 8px 0; color: #0073aa; display: flex; align-items: center;">';
            echo '<span style="margin-right: 8px; color: #ee8208;">⭐</span> <strong>Хотите все функции?</strong>';
            echo '</p>';
            echo '<p style="margin: 8px 0 0 0; color: #555; font-size: 13px;">';
            echo 'Обновитесь до <a href="' . admin_url('options-general.php?page=ok-auto-post&tab=license') . '" style="color: #ee8208; font-weight: 500;">PRO версии</a> и получите доступ ко всем переменным и дополнительным функциям!';
            echo '</p>';
            echo '</div>';
        }
    }
    
    /**
     * Валидация настроек
     */
    public function validate_options($input) {
        $output = $this->options ?? array();
        
        $output['app_id'] = sanitize_text_field($input['app_id'] ?? '');
        $output['app_key'] = sanitize_text_field($input['app_key'] ?? '');
        $output['app_secret_key'] = sanitize_text_field($input['app_secret_key'] ?? '');
        $output['access_token'] = sanitize_text_field($input['access_token'] ?? '');
        
        $group_ids = sanitize_text_field($input['group_ids'] ?? '');
        $output['group_ids'] = $group_ids;
        
        $group_ids_array = array();
        if (!empty($group_ids)) {
            $group_ids_array = array_map('trim', explode(',', $group_ids));
            $group_ids_array = array_filter($group_ids_array, function($id) {
                return is_numeric($id) && !empty($id);
            });
        }
        $output['group_ids_array'] = $group_ids_array;
        
        $output['post_types'] = array_map('sanitize_text_field', $input['post_types'] ?? array('post'));
        $output['post_template'] = wp_kses_post($input['post_template'] ?? '');
        
        $output['auto_posting_enabled'] = isset($input['auto_posting_enabled']) ? (bool)$input['auto_posting_enabled'] : true;
        
        return $output;
    }
    
    /**
     * AJAX переключение автопостинга
     */
    public function ajax_toggle_auto_posting() {
        check_ajax_referer('ok_toggle_auto_posting', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Недостаточно прав');
        }
        
        try {
            $enable = isset($_POST['enable']) ? (bool)$_POST['enable'] : false;
            $this->core->toggle_auto_posting($enable);
            
            $message = $enable ? '✅ Автопостинг включен' : '⏸️ Автопостинг отключен';
            
            wp_send_json_success(array(
                'message' => $message,
                'enabled' => $enable,
                'button_text' => $enable ? 'Выключить автопостинг' : 'Включить автопостинг',
                'status_text' => $enable ? '✅ Включен' : '⏸️ Отключен',
                'status_class' => $enable ? 'status-enabled' : 'status-disabled'
            ));
            
        } catch (Exception $e) {
            wp_send_json_error('Ошибка: ' . $e->getMessage());
        }
    }
    
    /**
     * AJAX проверка подключения
     */
    public function ajax_test_connection() {
        check_ajax_referer('ok_test_connection', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Недостаточно прав');
        }
        
        $credentials = array(
            'app_id' => sanitize_text_field($_POST['app_id'] ?? ''),
            'app_key' => sanitize_text_field($_POST['app_key'] ?? ''),
            'app_secret_key' => sanitize_text_field($_POST['app_secret_key'] ?? ''),
            'access_token' => sanitize_text_field($_POST['access_token'] ?? '')
        );
        
        $errors = array();
        
        if (empty($credentials['app_id'])) {
            $errors[] = 'Application ID обязателен';
        }
        
        if (empty($credentials['app_key'])) {
            $errors[] = 'Публичный ключ обязателен';
        }
        
        if (empty($credentials['app_secret_key'])) {
            $errors[] = 'Секретный ключ обязателен';
        }
        
        if (empty($credentials['access_token'])) {
            $errors[] = 'Access token обязателен';
        }
        
        if (!empty($errors)) {
            wp_send_json_error('Ошибки валидации: ' . implode(', ', $errors));
        }
        
        try {
            $api = new OkAutoPostAPI($credentials);
            $user_info = $api->test_api_connection($credentials);
            wp_send_json_success(array(
                'message' => '✅ Успешное подключение! Пользователь: ' . $user_info['name'],
                'user' => $user_info
            ));
        } catch (Exception $e) {
            wp_send_json_error($e->getMessage());
        }
    }
    
    /**
     * AJAX отправка тестового поста
     */
    public function ajax_test_post() {
        check_ajax_referer('ok_test_post', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Недостаточно прав');
        }
        
        $group_ids = $this->options['group_ids_array'] ?? array();
        if (empty($group_ids)) {
            wp_send_json_error('Не указаны ID групп для тестирования');
        }
        
        try {
            $api = new OkAutoPostAPI($this->options);
            $test_message = $api->prepare_test_message();
            
            $results = array();
            $success_count = 0;
            $error_count = 0;
            
            foreach ($group_ids as $group_id) {
                try {
                    $result = $api->post_to_group($group_id, $test_message);
                    $results[$group_id] = array('success' => true, 'data' => $result);
                    $success_count++;
                    
                    OkAutoPostLogger::log_post(0, $group_id, 'success', 'Тестовый пост опубликован');
                    
                } catch (Exception $e) {
                    $results[$group_id] = array('success' => false, 'error' => $e->getMessage());
                    $error_count++;
                    
                    OkAutoPostLogger::log_post(0, $group_id, 'error', $e->getMessage());
                }
            }
            
            $message = "Тестовая публикация завершена: ";
            $message .= "✅ Успешно: {$success_count}, ";
            $message .= "❌ Ошибок: {$error_count}";
            
            wp_send_json_success(array(
                'message' => $message,
                'results' => $results
            ));
            
        } catch (Exception $e) {
            wp_send_json_error('Ошибка при отправке тестового поста: ' . $e->getMessage());
        }
    }
    
    /**
     * AJAX перепубликация поста
     */
    public function ajax_republish_post() {
        check_ajax_referer('ok_republish_post', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Недостаточно прав');
        }
        
        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        
        if (!$post_id) {
            wp_send_json_error('Не указан ID поста');
        }
        
        try {
            delete_post_meta($post_id, '_ok_auto_posted');
            $this->core->delayed_auto_post_to_ok($post_id);
            
            wp_send_json_success('Пост отправлен на перепубликацию');
            
        } catch (Exception $e) {
            wp_send_json_error('Ошибка: ' . $e->getMessage());
        }
    }
    
    /**
     * AJAX очистка метки публикации у поста
     */
    public function ajax_clear_post_meta() {
        check_ajax_referer('ok_clear_post_meta', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Недостаточно прав');
        }
        
        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        
        if (!$post_id) {
            wp_send_json_error('Не указан ID поста');
        }
        
        try {
            delete_post_meta($post_id, '_ok_auto_posted');
            wp_send_json_success('Метка публикации удалена. Пост можно отправить снова.');
            
        } catch (Exception $e) {
            wp_send_json_error('Ошибка: ' . $e->getMessage());
        }
    }
    
    /**
     * AJAX очистка ВСЕХ меток публикации
     */
    public function ajax_clear_all_meta() {
        check_ajax_referer('ok_clear_all_meta', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Недостаточно прав');
        }
        
        try {
            global $wpdb;
            
            $deleted_count = $wpdb->query(
                $wpdb->prepare(
                    "DELETE FROM {$wpdb->postmeta} WHERE meta_key = %s",
                    '_ok_auto_posted'
                )
            );
            
            $wpdb->query(
                $wpdb->prepare(
                    "DELETE FROM {$wpdb->options} WHERE option_name LIKE %s",
                    '_transient_ok_posting_lock_%'
                )
            );
            
            wp_send_json_success('Удалено ' . $deleted_count . ' меток публикации. Все посты можно отправить заново.');
            
        } catch (Exception $e) {
            wp_send_json_error('Ошибка: ' . $e->getMessage());
        }
    }
    
    /**
     * AJAX очистка логов отладки
     */
    public function ajax_clear_debug_logs() {
        check_ajax_referer('ok_clear_debug_logs', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Недостаточно прав');
        }
        
        OkAutoPostLogger::clear_debug_logs();
        wp_send_json_success('Логи отладки очищены');
    }
    
    /**
     * AJAX очистка логов постов
     */
    public function ajax_clear_post_logs() {
        check_ajax_referer('ok_clear_post_logs', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Недостаточно прав');
        }
        
        OkAutoPostLogger::clear_post_logs();
        wp_send_json_success('Логи постов очищены');
    }
    
    /**
     * AJAX очистка всех данных
     */
    public function ajax_cleanup_all_data() {
        check_ajax_referer('ok_cleanup_all_data', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Недостаточно прав');
        }
        
        try {
            $this->core->cleanup_plugin_data();
            wp_send_json_success('Все данные плагина успешно очищены');
        } catch (Exception $e) {
            wp_send_json_error('Ошибка при очистке: ' . $e->getMessage());
        }
    }
    
    /**
     * Навигация по вкладкам
     */
    public function show_tab_navigation() {
        $tabs = array(
            'settings' => 'Настройки',
            'license' => 'Лицензия PRO',
            'logs' => 'Статус постов',
            'debug' => 'Отладка'
        );
        
        $current_tab = $this->current_tab;
        
        echo '<h2 class="nav-tab-wrapper">';
        foreach ($tabs as $tab => $name) {
            $class = ($tab == $current_tab) ? ' nav-tab-active' : '';
            $url = add_query_arg('tab', $tab, admin_url('options-general.php?page=ok-auto-post'));
            echo '<a href="' . esc_url($url) . '" class="nav-tab' . $class . '">' . esc_html($name) . '</a>';
        }
        echo '</h2>';
    }
    
    /**
     * Показать инструменты очистки
     */
    public function show_cleanup_tools() {
        echo '<div class="card" style="margin-top: 20px;">';
        echo '<h3>Очистка данных</h3>';
        echo '<p><strong>Внимание:</strong> Эти действия нельзя отменить! Используйте осторожно.</p>';
        
        echo '<table class="widefat">';
        echo '<tr>';
        echo '<td><strong>Очистить логи постов:</strong></td>';
        echo '<td><button type="button" id="ok-clear-post-logs" class="button">Очистить логи</button></td>';
        echo '</tr>';
        echo '<tr>';
        echo '<td><strong>Очистить логи отладки:</strong></td>';
        echo '<td><button type="button" id="ok-clear-debug-logs" class="button">Очистить логи</button></td>';
        echo '</tr>';
        echo '<tr>';
        echo '<td><strong>Очистить ВСЕ метки публикации:</strong></td>';
        echo '<td><button type="button" id="ok-clear-all-meta" class="button button-warning">Очистить все метки</button></td>';
        echo '</tr>';
        echo '<tr>';
        echo '<td><strong>Очистить ВСЕ данные плагина:</strong></td>';
        echo '<td><button type="button" id="ok-cleanup-all" class="button button-danger">Полная очистка</button></td>';
        echo '</tr>';
        echo '</table>';
        
        echo '<div id="ok-cleanup-result" style="margin-top: 10px;"></div>';
        echo '</div>';
    }
    
    /**
     * Показать уведомления в админке
     */
    public function show_admin_notices() {
        // Показываем сохраненные уведомления
        foreach ($this->admin_notices as $notice) {
            echo '<div class="notice notice-' . esc_attr($notice['type']) . ' is-dismissible">';
            echo '<p>' . esc_html($notice['message']) . '</p>';
            echo '</div>';
        }
        
        // Очищаем уведомления после показа
        $this->admin_notices = array();
        
        // Также показываем стандартные уведомления WordPress
        settings_errors('ok_auto_post_options');
    }
    
    /**
     * Страница настроек
     */
    public function settings_page() {
        // Подключаем шаблон
        include OK_AUTO_POST_PLUGIN_PATH . 'templates/admin-settings.php';
    }
}